import React, { useState } from "react";
import { useLanguage } from "@/contexts/LanguageContext";
import { Technician, CostAnalysisData } from "@/types/analysis";

interface CostsTabProps {
  costData: CostAnalysisData;
  technicians: Technician[];
  selectedTechnician: string;
  setSelectedTechnician: (value: string) => void;
  startDate: string;
  setStartDate: (value: string) => void;
  endDate: string;
  setEndDate: (value: string) => void;
  onFilter: () => void;
}

export default function CostsTab({
  costData,
  technicians,
  selectedTechnician,
  setSelectedTechnician,
  startDate,
  setStartDate,
  endDate,
  setEndDate,
  onFilter,
}: CostsTabProps) {
  const { t } = useLanguage();
  const [editingRate, setEditingRate] = useState<string | null>(null);
  const [newRate, setNewRate] = useState("");

  const handleUpdateRate = async (technicianId: string) => {
    if (!newRate || isNaN(parseFloat(newRate))) return;

    try {
      const response = await fetch("/api/admin/technicians/pricing", {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          technicianId,
          hourlyRate: parseFloat(newRate),
        }),
      });

      if (response.ok) {
        // Refresh data by calling the parent's fetch function
        window.location.reload();
        setEditingRate(null);
        setNewRate("");
      }
    } catch (error) {
      console.error("Error updating rate:", error);
    }
  };

  const filteredTechnicians = selectedTechnician
    ? technicians.filter((tech) => tech.id === selectedTechnician)
    : technicians;

  return (
    <>
      {/* Filters - At the top */}
      <div className='bg-white p-4 sm:p-6 rounded-lg shadow mb-6 sm:mb-8'>
        <h2 className='text-base sm:text-lg font-semibold mb-4'>
          {t("tickets.filters")}
        </h2>
        <div className='grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4'>
          <div>
            <label className='block text-sm font-medium text-gray-700 mb-1'>
              {t("admin.technicians")}
            </label>
            <select
              value={selectedTechnician}
              onChange={(e) => setSelectedTechnician(e.target.value)}
              className='w-full border border-gray-300 rounded-md px-3 py-2 text-sm sm:text-base'
            >
              <option value=''>
                {t("admin.allTechnicians") || "All Technicians"}
              </option>
              {technicians.map((tech) => {
                const displayName = tech.name || tech.email;
                const externalTag = tech.isExternal ? " [External]" : "";
                const companyInfo = tech.companyName ? ` (${tech.companyName})` : "";
                return (
                  <option key={tech.id} value={tech.id}>
                    {displayName}{externalTag}{companyInfo}
                  </option>
                );
              })}
            </select>
          </div>
          <div>
            <label className='block text-sm font-medium text-gray-700 mb-1'>
              {t("tickets.startDate")}
            </label>
            <input
              type='date'
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
              className='w-full border border-gray-300 rounded-md px-3 py-2 text-sm sm:text-base'
            />
          </div>
          <div>
            <label className='block text-sm font-medium text-gray-700 mb-1'>
              {t("tickets.endDate")}
            </label>
            <input
              type='date'
              value={endDate}
              onChange={(e) => setEndDate(e.target.value)}
              className='w-full border border-gray-300 rounded-md px-3 py-2 text-sm sm:text-base'
            />
          </div>
          <div className='flex items-end sm:col-span-2 lg:col-span-1'>
            <button
              onClick={onFilter}
              className='w-full bg-blue-600 text-white px-4 py-2 rounded-md hover:bg-blue-700 text-sm sm:text-base'
            >
              {t("tickets.applyFilters") || "Apply Filters"}
            </button>
          </div>
        </div>
      </div>

      {/* Cost Summary Cards */}
      <div className='grid grid-cols-2 sm:grid-cols-2 md:grid-cols-4 gap-3 sm:gap-6 mb-6 sm:mb-8'>
        <div className='bg-white p-4 sm:p-6 rounded-lg shadow'>
          <h3 className='text-xs sm:text-sm font-medium text-gray-500'>
            {t("analysis.totalTickets") || "Total Tickets"}
          </h3>
          <p className='text-xl sm:text-2xl font-bold text-gray-900'>
            {costData.summary.totalTickets}
          </p>
        </div>
        <div className='bg-white p-4 sm:p-6 rounded-lg shadow'>
          <h3 className='text-xs sm:text-sm font-medium text-gray-500'>
            {t("analysis.totalHours") || "Total Hours"}
          </h3>
          <p className='text-xl sm:text-2xl font-bold text-gray-900'>
            {costData.summary.totalHours}h
          </p>
        </div>
        <div className='bg-white p-4 sm:p-6 rounded-lg shadow'>
          <h3 className='text-xs sm:text-sm font-medium text-gray-500'>
            {t("analysis.totalCost") || "Total Cost"}
          </h3>
          <p className='text-xl sm:text-2xl font-bold text-gray-900'>
            €{costData.summary.totalCost}
          </p>
        </div>
        <div className='bg-white p-4 sm:p-6 rounded-lg shadow'>
          <h3 className='text-xs sm:text-sm font-medium text-gray-500'>
            {t("admin.technicians")}
          </h3>
          <p className='text-xl sm:text-2xl font-bold text-gray-900'>
            {costData.summary.technicianCount}
          </p>
        </div>
      </div>

      {/* Technician Hourly Rates */}
      <div className='bg-white rounded-lg shadow mb-6 sm:mb-8'>
        <div className='px-4 sm:px-6 py-3 sm:py-4 border-b border-gray-200'>
          <h2 className='text-base sm:text-lg font-semibold'>
            {t("analysis.technicianHourlyRates") || "Technician Hourly Rates"}
          </h2>
        </div>
        <div className='overflow-x-auto'>
          <table className='min-w-full divide-y divide-gray-200'>
            <thead className='bg-gray-50'>
              <tr>
                <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("admin.technicians")}
                </th>
                <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden sm:table-cell'>
                  {t("login.email")}
                </th>
                <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("analysis.hourlyRate") || "Rate"}
                </th>
                <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                  {t("common.actions")}
                </th>
              </tr>
            </thead>
            <tbody className='bg-white divide-y divide-gray-200'>
              {filteredTechnicians.map((tech) => (
                <tr key={tech.id}>
                  <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm font-medium text-gray-900'>
                    <div className='flex items-center gap-2 flex-wrap'>
                      <span>{tech.name || "No name"}</span>
                      {tech.isExternal && (
                        <span className='px-2 py-0.5 text-xs font-medium bg-blue-100 text-blue-800 rounded-full border border-blue-200'>
                          External
                        </span>
                      )}
                      {tech.companyName && (
                        <span className='px-2 py-0.5 text-xs font-medium bg-gray-100 text-gray-700 rounded-full border border-gray-200'>
                          {tech.companyName}
                        </span>
                      )}
                    </div>
                    <div className='text-xs text-gray-500 sm:hidden mt-1'>
                      {tech.email}
                    </div>
                  </td>
                  <td className='hidden sm:table-cell px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-500'>
                    {tech.email}
                  </td>
                  <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-900'>
                    {editingRate === tech.id ? (
                      <input
                        type='number'
                        value={newRate}
                        onChange={(e) => setNewRate(e.target.value)}
                        className='w-16 sm:w-20 border border-gray-300 rounded px-2 py-1 text-xs'
                        placeholder={tech.hourlyRate?.toString() || "0"}
                        step='0.01'
                        min='0'
                      />
                    ) : (
                      `€${tech.hourlyRate || 0}/h`
                    )}
                  </td>
                  <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-500'>
                    {editingRate === tech.id ? (
                      <div className='flex space-x-1 sm:space-x-2'>
                        <button
                          onClick={() => handleUpdateRate(tech.id)}
                          className='text-green-600 hover:text-green-900 text-xs'
                        >
                          {t("common.save") || "Save"}
                        </button>
                        <button
                          onClick={() => {
                            setEditingRate(null);
                            setNewRate("");
                          }}
                          className='text-gray-600 hover:text-gray-900 text-xs'
                        >
                          {t("common.cancel") || "Cancel"}
                        </button>
                      </div>
                    ) : (
                      <button
                        onClick={() => {
                          setEditingRate(tech.id);
                          setNewRate(tech.hourlyRate?.toString() || "");
                        }}
                        className='text-blue-600 hover:text-blue-900 text-xs'
                      >
                        {t("common.edit") || "Edit"}
                      </button>
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Cost Analysis by Technician */}
      {costData && costData.technicians.length > 0 && (
        <div className='bg-white rounded-lg shadow'>
          <div className='px-4 sm:px-6 py-3 sm:py-4 border-b border-gray-200'>
            <h2 className='text-base sm:text-lg font-semibold'>
              {t("analysis.paymentSummary") || "Payment Summary by Technician"}
            </h2>
          </div>
          <div className='overflow-x-auto'>
            <table className='min-w-full divide-y divide-gray-200'>
              <thead className='bg-gray-50'>
                <tr>
                  <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("admin.technicians")}
                  </th>
                  <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell'>
                    {t("analysis.hourlyRate") || "Hourly Rate"}
                  </th>
                  <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("tickets.title")}
                  </th>
                  <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.hours") || "Hours"}
                  </th>
                  <th className='px-3 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider'>
                    {t("analysis.totalCost") || "Total"}
                  </th>
                </tr>
              </thead>
              <tbody className='bg-white divide-y divide-gray-200'>
                {costData.technicians.map((tech) => (
                  <tr key={tech.technicianId}>
                    <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm font-medium text-gray-900'>
                      <div className='flex items-center gap-2 flex-wrap'>
                        <span>{tech.technicianName}</span>
                        {tech.isExternal && (
                          <span className='px-2 py-0.5 text-xs font-medium bg-blue-100 text-blue-800 rounded-full border border-blue-200'>
                            External
                          </span>
                        )}
                        {tech.companyName && (
                          <span className='px-2 py-0.5 text-xs font-medium bg-gray-100 text-gray-700 rounded-full border border-gray-200'>
                            {tech.companyName}
                          </span>
                        )}
                      </div>
                    </td>
                    <td className='hidden md:table-cell px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-500'>
                      €{tech.hourlyRate}/h
                    </td>
                    <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-500'>
                      {tech.totalTickets}
                    </td>
                    <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm text-gray-500'>
                      {tech.totalHours}h
                    </td>
                    <td className='px-3 sm:px-6 py-3 sm:py-4 text-xs sm:text-sm font-medium text-gray-900'>
                      €{tech.totalCost}
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        </div>
      )}
    </>
  );
}
